/*
 * Copyright (c) 2006-2021, RT-Thread Development Team
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author       Notes
 * 2025-05-11     tt       the first version
 */

#include <rtthread.h>

extern int ptc06_write(uint8_t* buf, size_t len);
extern size_t ptc06_read(uint8_t** buf);
extern size_t ptc06_read_image(uint8_t** buf, int length);
extern void open_uart();
extern void close_uart();

static int not_right_header(uint8_t* read_buf, size_t buf_len, uint8_t header[], size_t header_len)
{
    int i = 0;
    for (i = 0; i < buf_len && i < header_len; i++)
    {
        if (read_buf[i] != header[i])
        {
            break;
        }
    }
    if (i == header_len)
    {
        return 0;
    }
    else
    {
        return 1;
    }
}
int reset_version_clear()
{
    open_uart();
    uint8_t cmd_buf[] = { 0x56, 0x00, 0x36, 0x01, 0x03 };
    if (ptc06_write(cmd_buf, sizeof(cmd_buf) / sizeof(uint8_t)))
    {
        close_uart();
        return 1;
    }
    uint8_t* reader_buf = NULL;
    size_t size = ptc06_read(&reader_buf);
    rt_free(reader_buf);
    close_uart();
    return 0;
}
int reset()
{
    open_uart();
    uint8_t buf[4] = { 0x56, 0x00, 0x26, 0x00 };
    if (ptc06_write(buf, 4))
    {
        close_uart();
        return 1;
    }

    uint8_t* reader_buf = NULL;
    size_t size = ptc06_read(&reader_buf);
    uint8_t right_header[] = { 0x76, 0x00, 0x26, 0x00 };
    close_uart();
    if (not_right_header(reader_buf, size, right_header, 4))
    {
        rt_free(reader_buf);
        return 1;
    }
    else
    {
        rt_free(reader_buf);
        // 去除reset之后会返回的版本数据
        reset_version_clear();
        return 0;
    }
}

int take_photo()
{
    open_uart();
    uint8_t cmd_buf[5] = { 0x56, 0x00, 0x36, 0x01, 0x00 };
    uint8_t right_header[] = { 0x76, 0x00, 0x36, 0x00, 0x00 };
    if (ptc06_write(cmd_buf, 5))
    {
        close_uart();
        return 1;
    }
    uint8_t* reader_buf = NULL;
    size_t size = ptc06_read(&reader_buf);
    close_uart();
    int ret = not_right_header(reader_buf, size, right_header, 5);

    rt_free(reader_buf);
    return ret;
}

int get_photo_size(uint16_t *photo_size)
{
    open_uart();
    uint8_t cmd_buf[] = { 0x56, 0x00, 0x34, 0x01, 0x00 };
    uint8_t right_header[] = { 0x76, 0x00, 0x34, 0x00, 0x04, 0x00, 0x00 };
    if (ptc06_write(cmd_buf, sizeof(cmd_buf) / sizeof(uint8_t)))
    {
        close_uart();
        return 1;
    }
    uint8_t* reader_buf = NULL;
    size_t size = ptc06_read(&reader_buf);
    close_uart();
    if (not_right_header(reader_buf, size, right_header, sizeof(right_header) / sizeof(uint8_t)))
    {
        rt_free(reader_buf);
        return 1;
    }
    *photo_size = 0;
    *photo_size |= reader_buf[7] << 8;
    *photo_size |= reader_buf[8];
    rt_free(reader_buf);
    return 0;
}

int get_all_data(uint16_t photo_size)
{
    open_uart();
    uint8_t right_header[] = { 0x76, 0x00, 0x32, 0x00, 0x00 };
    uint8_t cmd_buf[] = { 0x56, 0x00, 0x32, 0x0c, 0x00, 0x0a, 0, 0, 0x00, 0x00, // 图片数据起始地址
            0, 0, ((uint8_t) (photo_size >> 8)), ((uint8_t) photo_size), // 本次读取长度
            0x00, 0xff // 结尾
            };
    if (ptc06_write(cmd_buf, sizeof(cmd_buf) / sizeof(uint8_t)))
    {
        close_uart();
        return 1;
    }
    uint8_t* photo_buf = (uint8_t*) rt_malloc(photo_size + 10);
    size_t size = ptc06_read_image(&photo_buf, photo_size + 10);
    close_uart();
    if (not_right_header(photo_buf, size, right_header, 5)
            || not_right_header(&photo_buf[5 + photo_size], size - 5 - photo_size, right_header, 5))
    {
        rt_free(photo_buf);
        return 1;
    }
    int part_size = 48;
    rt_kprintf("data = [");
    rt_kprintf("%02x", photo_buf[5]);
    for (size_t i = 6; i < photo_size + 5; ++i)
    {
        rt_kprintf(", %02x", photo_buf[i]);
        if ((i - 5) % part_size == 0)
        {
            rt_kprintf("\n");
        }
    }
    rt_kprintf("]\n");
    rt_free(photo_buf);
    rt_thread_mdelay(10);
    return 0;
}

//int get_photo_data(uint16_t photo_size, uint8_t *photo_buf)
//{
//    open_uart();
//    uint8_t right_header[] = { 0x76, 0x00, 0x32, 0x00, 0x00 };
//    uint8_t cmd_buf[] = { 0x56, 0x00, 0x32, 0x0c, 0x00, 0x0a, 0, 0, 0x00, 0x00, // 图片数据起始地址
//            0, 0, ((uint8_t) (photo_size >> 8)), ((uint8_t) photo_size), // 本次读取长度
//            0x00, 0xff // 结尾
//            };
//    int s, l;
//    int part_size = (RT_SERIAL_RB_BUFSZ - 10) / 8 * 8;
//    uint8_t * buf;
////    rt_kprintf("data = [");
//    for (int i = 0; i < (photo_size / part_size) + 1; i++)
//    {
////        open_uart();
//        s = i * part_size;
//        l = part_size;
//        if (s >= photo_size)
//        {
//            break;
//        }
//        if (s + l >= photo_size)
//        {
//            l = photo_size - s;
//        }
//        cmd_buf[8] = (uint8_t) (s >> 8);
//        cmd_buf[9] = (uint8_t) s;
//        cmd_buf[12] = (uint8_t) (l >> 8);
//        cmd_buf[13] = (uint8_t) l;
//        if (ptc06_write(cmd_buf, sizeof(cmd_buf) / sizeof(uint8_t)))
//        {
//            close_uart();
//            return 1;
//        }
//        size_t size = ptc06_read(&buf);
//        // 读取到头部信息
//        if (not_right_header(buf, size, right_header, 5)
//                || not_right_header(&buf[5 + part_size], size - 5 - part_size, right_header, 5))
//        {
//            i -= 1;
//            continue;
//        }
////        for (int j = s; j < s + l; j++)
////        {
////            if (j == 0)
////            {
////                rt_kprintf("%02x", buf[j - s + 5]);
////            }
////            else
////            {
////                rt_kprintf(",%02x", buf[j - s + 5]);
////            }
////        }
////        rt_kprintf("\n");
//        for (int j = s; j < s + l; j++)
//        {
//            photo_buf[j] = buf[j - s + 5];
//        }
////        close_uart();
//    }
//    rt_kprintf("]\n");
//
//    rt_kprintf("data = [");
//    rt_kprintf("%02x", photo_buf[5]);
//    for (size_t i = 6; i < photo_size + 5; ++i)
//    {
//        rt_kprintf(", %02x", photo_buf[i]);
//        if ((i - 5) % part_size == 0)
//        {
//            rt_kprintf("\n");
//        }
//    }
//    rt_kprintf("]\n");
//    return 0;
//}

int clear_photo_cache()
{
    open_uart();
    uint8_t cmd_buf[] = { 0x56, 0x00, 0x36, 0x01, 0x03 };
    uint8_t right_header[] = { 0x76, 0x00, 0x36, 0x00, 0x00 };
    if (ptc06_write(cmd_buf, sizeof(cmd_buf) / sizeof(uint8_t)))
    {
        close_uart();
        return 1;
    }
    uint8_t* reader_buf = NULL;
    size_t size = ptc06_read(&reader_buf);
    close_uart();
    if (not_right_header(reader_buf, size, right_header, sizeof(right_header) / sizeof(uint8_t)))
    {
        rt_free(reader_buf);
        return 1;
    }
    rt_free(reader_buf);
    return 0;
}
int ptc_test()
{
    rt_thread_mdelay(2500);
    rt_kprintf("start ptc-06\n");
    if (reset())
    {
        rt_kprintf("reset fail\n");
    }
    while (1)
    {
        rt_thread_mdelay(500);
        if (take_photo())
        {
            rt_kprintf("take photo fail\n");
            continue;
        }
        rt_thread_mdelay(2000);
        uint16_t photo_size;
        if (get_photo_size(&photo_size))
        {
            rt_kprintf("get photo size fail\n");
            continue;
        }
        rt_thread_mdelay(1000);
        rt_kprintf("photo size: %d\n", photo_size);
//        uint8_t* photo_data = rt_malloc(photo_size); //图片数据
//        if (get_photo_data(photo_size, photo_data))
        if (get_all_data(photo_size))
        {
            rt_kprintf("get photo data fail\n");
            continue;
        }
        if (clear_photo_cache())
        {
            rt_kprintf("clear cache fail\n");
            continue;
        }
    }
}

INIT_APP_EXPORT(ptc_test);
