/***************************************************************************//**
* \file cyip_sdhc.h
*
* \brief
* SDHC IP definitions
*
********************************************************************************
* \copyright
* (c) (2016-2022), Cypress Semiconductor Corporation (an Infineon company) or
* an affiliate of Cypress Semiconductor Corporation.
*
* SPDX-License-Identifier: Apache-2.0
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*******************************************************************************/

#ifndef _CYIP_SDHC_H_
#define _CYIP_SDHC_H_

#include "cyip_headers.h"

/*******************************************************************************
*                                     SDHC
*******************************************************************************/

#define SDHC_WRAP_SECTION_SIZE                  0x00000020UL
#define SDHC_CORE_SECTION_SIZE                  0x00001000UL
#define SDHC_SECTION_SIZE                       0x00010000UL

/**
  * \brief MMIO at SDHC wrapper level (SDHC_WRAP)
  */
typedef struct {
  __IOM uint32_t CTL;                           /*!< 0x00000000 Top level wrapper control */
   __IM uint32_t RESERVED[7];
} SDHC_WRAP_V1_Type;                            /*!< Size = 32 (0x20) */

/**
  * \brief MMIO for Synopsys Mobile Storage Host Controller IP (SDHC_CORE)
  */
typedef struct {
  __IOM uint32_t SDMASA_R;                      /*!< 0x00000000 SDMA System Address register */
  __IOM uint16_t BLOCKSIZE_R;                   /*!< 0x00000004 Block Size register */
  __IOM uint16_t BLOCKCOUNT_R;                  /*!< 0x00000006 16-bit Block Count register */
  __IOM uint32_t ARGUMENT_R;                    /*!< 0x00000008 Argument register */
  __IOM uint16_t XFER_MODE_R;                   /*!< 0x0000000C Transfer Mode register */
  __IOM uint16_t CMD_R;                         /*!< 0x0000000E Command register */
   __IM uint32_t RESP01_R;                      /*!< 0x00000010 Response Register 0/1 */
   __IM uint32_t RESP23_R;                      /*!< 0x00000014 Response Register 2/3 */
   __IM uint32_t RESP45_R;                      /*!< 0x00000018 Response Register 4/5 */
   __IM uint32_t RESP67_R;                      /*!< 0x0000001C Response Register 6/7 */
  __IOM uint32_t BUF_DATA_R;                    /*!< 0x00000020 Buffer Data Port Register */
   __IM uint32_t PSTATE_REG;                    /*!< 0x00000024 Present State Register */
  __IOM uint8_t  HOST_CTRL1_R;                  /*!< 0x00000028 Host Control 1 Register */
  __IOM uint8_t  PWR_CTRL_R;                    /*!< 0x00000029 Power Control Register */
  __IOM uint8_t  BGAP_CTRL_R;                   /*!< 0x0000002A Block Gap Control Register */
  __IOM uint8_t  WUP_CTRL_R;                    /*!< 0x0000002B Wakeup Control Register */
  __IOM uint16_t CLK_CTRL_R;                    /*!< 0x0000002C Clock Control Register */
  __IOM uint8_t  TOUT_CTRL_R;                   /*!< 0x0000002E Timeout Control Register */
  __IOM uint8_t  SW_RST_R;                      /*!< 0x0000002F Software Reset Register */
  __IOM uint16_t NORMAL_INT_STAT_R;             /*!< 0x00000030 Normal Interrupt Status Register */
  __IOM uint16_t ERROR_INT_STAT_R;              /*!< 0x00000032 Error Interrupt Status Register */
  __IOM uint16_t NORMAL_INT_STAT_EN_R;          /*!< 0x00000034 Normal Interrupt Status Enable Register */
  __IOM uint16_t ERROR_INT_STAT_EN_R;           /*!< 0x00000036 Error Interrupt Status Enable Register */
  __IOM uint16_t NORMAL_INT_SIGNAL_EN_R;        /*!< 0x00000038 Normal Interrupt Signal Enable Register */
  __IOM uint16_t ERROR_INT_SIGNAL_EN_R;         /*!< 0x0000003A Error Interrupt Signal Enable Register */
   __IM uint16_t AUTO_CMD_STAT_R;               /*!< 0x0000003C Auto CMD Status Register */
  __IOM uint16_t HOST_CTRL2_R;                  /*!< 0x0000003E Host Control 2 Register */
   __IM uint32_t CAPABILITIES1_R;               /*!< 0x00000040 Capabilities 1 Register - 0 to 31 */
   __IM uint32_t CAPABILITIES2_R;               /*!< 0x00000044 Capabilities Register - 32 to 63 */
   __IM uint32_t CURR_CAPABILITIES1_R;          /*!< 0x00000048 Current Capabilities Register - 0 to 31 */
   __IM uint32_t CURR_CAPABILITIES2_R;          /*!< 0x0000004C Maximum Current Capabilities Register - 32 to 63 */
   __OM uint16_t FORCE_AUTO_CMD_STAT_R;         /*!< 0x00000050 Force Event Register for Auto CMD Error Status register */
  __IOM uint16_t FORCE_ERROR_INT_STAT_R;        /*!< 0x00000052 Force Event Register for Error Interrupt Status */
   __IM uint8_t  ADMA_ERR_STAT_R;               /*!< 0x00000054 ADMA Error Status Register */
   __IM uint8_t  RESERVED[3];
  __IOM uint32_t ADMA_SA_LOW_R;                 /*!< 0x00000058 ADMA System Address Register - Low */
   __IM uint32_t RESERVED1[7];
  __IOM uint32_t ADMA_ID_LOW_R;                 /*!< 0x00000078 ADMA3 Integrated Descriptor Address Register - Low */
   __IM uint16_t RESERVED2[65];
   __IM uint16_t HOST_CNTRL_VERS_R;             /*!< 0x000000FE Host Controller Version */
   __IM uint32_t RESERVED3[32];
   __IM uint32_t CQVER;                         /*!< 0x00000180 Command Queuing Version register */
   __IM uint32_t CQCAP;                         /*!< 0x00000184 Command Queuing Capabilities register */
  __IOM uint32_t CQCFG;                         /*!< 0x00000188 Command Queuing Configuration register */
  __IOM uint32_t CQCTL;                         /*!< 0x0000018C Command Queuing Control register */
  __IOM uint32_t CQIS;                          /*!< 0x00000190 Command Queuing Interrupt Status register */
  __IOM uint32_t CQISE;                         /*!< 0x00000194 Command Queuing Interrupt Status Enable register */
  __IOM uint32_t CQISGE;                        /*!< 0x00000198 Command Queuing Interrupt signal enable register */
  __IOM uint32_t CQIC;                          /*!< 0x0000019C Command Queuing Interrupt Coalescing register */
  __IOM uint32_t CQTDLBA;                       /*!< 0x000001A0 Command Queuing Task Descriptor List Base Address register */
   __IM uint32_t RESERVED4;
  __IOM uint32_t CQTDBR;                        /*!< 0x000001A8 Command Queuing DoorBell register */
  __IOM uint32_t CQTCN;                         /*!< 0x000001AC Command Queuing TaskClear Notification register */
   __IM uint32_t CQDQS;                         /*!< 0x000001B0 Device queue status register */
   __IM uint32_t CQDPT;                         /*!< 0x000001B4 Device pending tasks register */
  __IOM uint32_t CQTCLR;                        /*!< 0x000001B8 Command Queuing DoorBell register */
   __IM uint32_t RESERVED5;
  __IOM uint32_t CQSSC1;                        /*!< 0x000001C0 CQ Send Status Configuration 1 register */
  __IOM uint32_t CQSSC2;                        /*!< 0x000001C4 CQ Send Status Configuration 2 register */
   __IM uint32_t CQCRDCT;                       /*!< 0x000001C8 Command response for direct command register */
   __IM uint32_t RESERVED6;
  __IOM uint32_t CQRMEM;                        /*!< 0x000001D0 Command response mode error mask register */
   __IM uint32_t CQTERRI;                       /*!< 0x000001D4 CQ Task Error Information register */
   __IM uint32_t CQCRI;                         /*!< 0x000001D8 CQ Command response index */
   __IM uint32_t CQCRA;                         /*!< 0x000001DC CQ Command response argument register */
   __IM uint32_t RESERVED7[200];
   __IM uint32_t MSHC_VER_ID_R;                 /*!< 0x00000500 MSHC version */
   __IM uint32_t MSHC_VER_TYPE_R;               /*!< 0x00000504 MSHC version type */
  __IOM uint8_t  MSHC_CTRL_R;                   /*!< 0x00000508 MSHC Control register */
   __IM uint8_t  RESERVED8[7];
  __IOM uint8_t  MBIU_CTRL_R;                   /*!< 0x00000510 MBIU Control register */
   __IM uint8_t  RESERVED9[27];
  __IOM uint16_t EMMC_CTRL_R;                   /*!< 0x0000052C eMMC Control register */
  __IOM uint16_t BOOT_CTRL_R;                   /*!< 0x0000052E eMMC Boot Control register */
   __IM uint32_t GP_IN_R;                       /*!< 0x00000530 General Purpose Input register */
  __IOM uint32_t GP_OUT_R;                      /*!< 0x00000534 General Purpose Output register */
   __IM uint32_t RESERVED10[690];
} SDHC_CORE_V1_Type;                            /*!< Size = 4096 (0x1000) */

/**
  * \brief SD/eMMC Host Controller (SDHC)
  */
typedef struct {
        SDHC_WRAP_V1_Type WRAP;                 /*!< 0x00000000 MMIO at SDHC wrapper level */
   __IM uint32_t RESERVED[1016];
        SDHC_CORE_V1_Type CORE;                 /*!< 0x00001000 MMIO for Synopsys Mobile Storage Host Controller IP */
} SDHC_V1_Type;                                 /*!< Size = 8192 (0x2000) */


/* SDHC_WRAP.CTL */
#define SDHC_WRAP_CTL_ENABLE_Pos                31UL
#define SDHC_WRAP_CTL_ENABLE_Msk                0x80000000UL


/* SDHC_CORE.SDMASA_R */
#define SDHC_CORE_SDMASA_R_BLOCKCNT_SDMASA_Pos  0UL
#define SDHC_CORE_SDMASA_R_BLOCKCNT_SDMASA_Msk  0xFFFFFFFFUL
/* SDHC_CORE.BLOCKSIZE_R */
#define SDHC_CORE_BLOCKSIZE_R_XFER_BLOCK_SIZE_Pos 0UL
#define SDHC_CORE_BLOCKSIZE_R_XFER_BLOCK_SIZE_Msk 0xFFFUL
#define SDHC_CORE_BLOCKSIZE_R_SDMA_BUF_BDARY_Pos 12UL
#define SDHC_CORE_BLOCKSIZE_R_SDMA_BUF_BDARY_Msk 0x7000UL
/* SDHC_CORE.BLOCKCOUNT_R */
#define SDHC_CORE_BLOCKCOUNT_R_BLOCK_CNT_Pos    0UL
#define SDHC_CORE_BLOCKCOUNT_R_BLOCK_CNT_Msk    0xFFFFUL
/* SDHC_CORE.ARGUMENT_R */
#define SDHC_CORE_ARGUMENT_R_ARGUMENT_Pos       0UL
#define SDHC_CORE_ARGUMENT_R_ARGUMENT_Msk       0xFFFFFFFFUL
/* SDHC_CORE.XFER_MODE_R */
#define SDHC_CORE_XFER_MODE_R_DMA_ENABLE_Pos    0UL
#define SDHC_CORE_XFER_MODE_R_DMA_ENABLE_Msk    0x1UL
#define SDHC_CORE_XFER_MODE_R_BLOCK_COUNT_ENABLE_Pos 1UL
#define SDHC_CORE_XFER_MODE_R_BLOCK_COUNT_ENABLE_Msk 0x2UL
#define SDHC_CORE_XFER_MODE_R_AUTO_CMD_ENABLE_Pos 2UL
#define SDHC_CORE_XFER_MODE_R_AUTO_CMD_ENABLE_Msk 0xCUL
#define SDHC_CORE_XFER_MODE_R_DATA_XFER_DIR_Pos 4UL
#define SDHC_CORE_XFER_MODE_R_DATA_XFER_DIR_Msk 0x10UL
#define SDHC_CORE_XFER_MODE_R_MULTI_BLK_SEL_Pos 5UL
#define SDHC_CORE_XFER_MODE_R_MULTI_BLK_SEL_Msk 0x20UL
#define SDHC_CORE_XFER_MODE_R_RESP_TYPE_Pos     6UL
#define SDHC_CORE_XFER_MODE_R_RESP_TYPE_Msk     0x40UL
#define SDHC_CORE_XFER_MODE_R_RESP_ERR_CHK_ENABLE_Pos 7UL
#define SDHC_CORE_XFER_MODE_R_RESP_ERR_CHK_ENABLE_Msk 0x80UL
#define SDHC_CORE_XFER_MODE_R_RESP_INT_DISABLE_Pos 8UL
#define SDHC_CORE_XFER_MODE_R_RESP_INT_DISABLE_Msk 0x100UL
/* SDHC_CORE.CMD_R */
#define SDHC_CORE_CMD_R_RESP_TYPE_SELECT_Pos    0UL
#define SDHC_CORE_CMD_R_RESP_TYPE_SELECT_Msk    0x3UL
#define SDHC_CORE_CMD_R_SUB_CMD_FLAG_Pos        2UL
#define SDHC_CORE_CMD_R_SUB_CMD_FLAG_Msk        0x4UL
#define SDHC_CORE_CMD_R_CMD_CRC_CHK_ENABLE_Pos  3UL
#define SDHC_CORE_CMD_R_CMD_CRC_CHK_ENABLE_Msk  0x8UL
#define SDHC_CORE_CMD_R_CMD_IDX_CHK_ENABLE_Pos  4UL
#define SDHC_CORE_CMD_R_CMD_IDX_CHK_ENABLE_Msk  0x10UL
#define SDHC_CORE_CMD_R_DATA_PRESENT_SEL_Pos    5UL
#define SDHC_CORE_CMD_R_DATA_PRESENT_SEL_Msk    0x20UL
#define SDHC_CORE_CMD_R_CMD_TYPE_Pos            6UL
#define SDHC_CORE_CMD_R_CMD_TYPE_Msk            0xC0UL
#define SDHC_CORE_CMD_R_CMD_INDEX_Pos           8UL
#define SDHC_CORE_CMD_R_CMD_INDEX_Msk           0x3F00UL
/* SDHC_CORE.RESP01_R */
#define SDHC_CORE_RESP01_R_RESP01_Pos           0UL
#define SDHC_CORE_RESP01_R_RESP01_Msk           0xFFFFFFFFUL
/* SDHC_CORE.RESP23_R */
#define SDHC_CORE_RESP23_R_RESP23_Pos           0UL
#define SDHC_CORE_RESP23_R_RESP23_Msk           0xFFFFFFFFUL
/* SDHC_CORE.RESP45_R */
#define SDHC_CORE_RESP45_R_RESP45_Pos           0UL
#define SDHC_CORE_RESP45_R_RESP45_Msk           0xFFFFFFFFUL
/* SDHC_CORE.RESP67_R */
#define SDHC_CORE_RESP67_R_RESP67_Pos           0UL
#define SDHC_CORE_RESP67_R_RESP67_Msk           0xFFFFFFFFUL
/* SDHC_CORE.BUF_DATA_R */
#define SDHC_CORE_BUF_DATA_R_BUF_DATA_Pos       0UL
#define SDHC_CORE_BUF_DATA_R_BUF_DATA_Msk       0xFFFFFFFFUL
/* SDHC_CORE.PSTATE_REG */
#define SDHC_CORE_PSTATE_REG_CMD_INHIBIT_Pos    0UL
#define SDHC_CORE_PSTATE_REG_CMD_INHIBIT_Msk    0x1UL
#define SDHC_CORE_PSTATE_REG_CMD_INHIBIT_DAT_Pos 1UL
#define SDHC_CORE_PSTATE_REG_CMD_INHIBIT_DAT_Msk 0x2UL
#define SDHC_CORE_PSTATE_REG_DAT_LINE_ACTIVE_Pos 2UL
#define SDHC_CORE_PSTATE_REG_DAT_LINE_ACTIVE_Msk 0x4UL
#define SDHC_CORE_PSTATE_REG_DAT_7_4_Pos        4UL
#define SDHC_CORE_PSTATE_REG_DAT_7_4_Msk        0xF0UL
#define SDHC_CORE_PSTATE_REG_WR_XFER_ACTIVE_Pos 8UL
#define SDHC_CORE_PSTATE_REG_WR_XFER_ACTIVE_Msk 0x100UL
#define SDHC_CORE_PSTATE_REG_RD_XFER_ACTIVE_Pos 9UL
#define SDHC_CORE_PSTATE_REG_RD_XFER_ACTIVE_Msk 0x200UL
#define SDHC_CORE_PSTATE_REG_BUF_WR_ENABLE_Pos  10UL
#define SDHC_CORE_PSTATE_REG_BUF_WR_ENABLE_Msk  0x400UL
#define SDHC_CORE_PSTATE_REG_BUF_RD_ENABLE_Pos  11UL
#define SDHC_CORE_PSTATE_REG_BUF_RD_ENABLE_Msk  0x800UL
#define SDHC_CORE_PSTATE_REG_CARD_INSERTED_Pos  16UL
#define SDHC_CORE_PSTATE_REG_CARD_INSERTED_Msk  0x10000UL
#define SDHC_CORE_PSTATE_REG_CARD_STABLE_Pos    17UL
#define SDHC_CORE_PSTATE_REG_CARD_STABLE_Msk    0x20000UL
#define SDHC_CORE_PSTATE_REG_CARD_DETECT_PIN_LEVEL_Pos 18UL
#define SDHC_CORE_PSTATE_REG_CARD_DETECT_PIN_LEVEL_Msk 0x40000UL
#define SDHC_CORE_PSTATE_REG_WR_PROTECT_SW_LVL_Pos 19UL
#define SDHC_CORE_PSTATE_REG_WR_PROTECT_SW_LVL_Msk 0x80000UL
#define SDHC_CORE_PSTATE_REG_DAT_3_0_Pos        20UL
#define SDHC_CORE_PSTATE_REG_DAT_3_0_Msk        0xF00000UL
#define SDHC_CORE_PSTATE_REG_CMD_LINE_LVL_Pos   24UL
#define SDHC_CORE_PSTATE_REG_CMD_LINE_LVL_Msk   0x1000000UL
#define SDHC_CORE_PSTATE_REG_HOST_REG_VOL_Pos   25UL
#define SDHC_CORE_PSTATE_REG_HOST_REG_VOL_Msk   0x2000000UL
#define SDHC_CORE_PSTATE_REG_CMD_ISSU_ERR_Pos   27UL
#define SDHC_CORE_PSTATE_REG_CMD_ISSU_ERR_Msk   0x8000000UL
#define SDHC_CORE_PSTATE_REG_SUB_CMD_STAT_Pos   28UL
#define SDHC_CORE_PSTATE_REG_SUB_CMD_STAT_Msk   0x10000000UL
/* SDHC_CORE.HOST_CTRL1_R */
#define SDHC_CORE_HOST_CTRL1_R_LED_CTRL_Pos     0UL
#define SDHC_CORE_HOST_CTRL1_R_LED_CTRL_Msk     0x1UL
#define SDHC_CORE_HOST_CTRL1_R_DAT_XFER_WIDTH_Pos 1UL
#define SDHC_CORE_HOST_CTRL1_R_DAT_XFER_WIDTH_Msk 0x2UL
#define SDHC_CORE_HOST_CTRL1_R_HIGH_SPEED_EN_Pos 2UL
#define SDHC_CORE_HOST_CTRL1_R_HIGH_SPEED_EN_Msk 0x4UL
#define SDHC_CORE_HOST_CTRL1_R_DMA_SEL_Pos      3UL
#define SDHC_CORE_HOST_CTRL1_R_DMA_SEL_Msk      0x18UL
#define SDHC_CORE_HOST_CTRL1_R_EXT_DAT_XFER_Pos 5UL
#define SDHC_CORE_HOST_CTRL1_R_EXT_DAT_XFER_Msk 0x20UL
#define SDHC_CORE_HOST_CTRL1_R_CARD_DETECT_TEST_LVL_Pos 6UL
#define SDHC_CORE_HOST_CTRL1_R_CARD_DETECT_TEST_LVL_Msk 0x40UL
#define SDHC_CORE_HOST_CTRL1_R_CARD_DETECT_SIG_SEL_Pos 7UL
#define SDHC_CORE_HOST_CTRL1_R_CARD_DETECT_SIG_SEL_Msk 0x80UL
/* SDHC_CORE.PWR_CTRL_R */
#define SDHC_CORE_PWR_CTRL_R_SD_BUS_PWR_VDD1_Pos 0UL
#define SDHC_CORE_PWR_CTRL_R_SD_BUS_PWR_VDD1_Msk 0x1UL
#define SDHC_CORE_PWR_CTRL_R_SD_BUS_VOL_VDD1_Pos 1UL
#define SDHC_CORE_PWR_CTRL_R_SD_BUS_VOL_VDD1_Msk 0xEUL
/* SDHC_CORE.BGAP_CTRL_R */
#define SDHC_CORE_BGAP_CTRL_R_STOP_BG_REQ_Pos   0UL
#define SDHC_CORE_BGAP_CTRL_R_STOP_BG_REQ_Msk   0x1UL
#define SDHC_CORE_BGAP_CTRL_R_CONTINUE_REQ_Pos  1UL
#define SDHC_CORE_BGAP_CTRL_R_CONTINUE_REQ_Msk  0x2UL
#define SDHC_CORE_BGAP_CTRL_R_RD_WAIT_CTRL_Pos  2UL
#define SDHC_CORE_BGAP_CTRL_R_RD_WAIT_CTRL_Msk  0x4UL
#define SDHC_CORE_BGAP_CTRL_R_INT_AT_BGAP_Pos   3UL
#define SDHC_CORE_BGAP_CTRL_R_INT_AT_BGAP_Msk   0x8UL
/* SDHC_CORE.WUP_CTRL_R */
#define SDHC_CORE_WUP_CTRL_R_WUP_CARD_INT_Pos   0UL
#define SDHC_CORE_WUP_CTRL_R_WUP_CARD_INT_Msk   0x1UL
#define SDHC_CORE_WUP_CTRL_R_WUP_CARD_INSERT_Pos 1UL
#define SDHC_CORE_WUP_CTRL_R_WUP_CARD_INSERT_Msk 0x2UL
#define SDHC_CORE_WUP_CTRL_R_WUP_CARD_REMOVAL_Pos 2UL
#define SDHC_CORE_WUP_CTRL_R_WUP_CARD_REMOVAL_Msk 0x4UL
/* SDHC_CORE.CLK_CTRL_R */
#define SDHC_CORE_CLK_CTRL_R_INTERNAL_CLK_EN_Pos 0UL
#define SDHC_CORE_CLK_CTRL_R_INTERNAL_CLK_EN_Msk 0x1UL
#define SDHC_CORE_CLK_CTRL_R_INTERNAL_CLK_STABLE_Pos 1UL
#define SDHC_CORE_CLK_CTRL_R_INTERNAL_CLK_STABLE_Msk 0x2UL
#define SDHC_CORE_CLK_CTRL_R_SD_CLK_EN_Pos      2UL
#define SDHC_CORE_CLK_CTRL_R_SD_CLK_EN_Msk      0x4UL
#define SDHC_CORE_CLK_CTRL_R_PLL_ENABLE_Pos     3UL
#define SDHC_CORE_CLK_CTRL_R_PLL_ENABLE_Msk     0x8UL
#define SDHC_CORE_CLK_CTRL_R_CLK_GEN_SELECT_Pos 5UL
#define SDHC_CORE_CLK_CTRL_R_CLK_GEN_SELECT_Msk 0x20UL
#define SDHC_CORE_CLK_CTRL_R_UPPER_FREQ_SEL_Pos 6UL
#define SDHC_CORE_CLK_CTRL_R_UPPER_FREQ_SEL_Msk 0xC0UL
#define SDHC_CORE_CLK_CTRL_R_FREQ_SEL_Pos       8UL
#define SDHC_CORE_CLK_CTRL_R_FREQ_SEL_Msk       0xFF00UL
/* SDHC_CORE.TOUT_CTRL_R */
#define SDHC_CORE_TOUT_CTRL_R_TOUT_CNT_Pos      0UL
#define SDHC_CORE_TOUT_CTRL_R_TOUT_CNT_Msk      0xFUL
/* SDHC_CORE.SW_RST_R */
#define SDHC_CORE_SW_RST_R_SW_RST_ALL_Pos       0UL
#define SDHC_CORE_SW_RST_R_SW_RST_ALL_Msk       0x1UL
#define SDHC_CORE_SW_RST_R_SW_RST_CMD_Pos       1UL
#define SDHC_CORE_SW_RST_R_SW_RST_CMD_Msk       0x2UL
#define SDHC_CORE_SW_RST_R_SW_RST_DAT_Pos       2UL
#define SDHC_CORE_SW_RST_R_SW_RST_DAT_Msk       0x4UL
/* SDHC_CORE.NORMAL_INT_STAT_R */
#define SDHC_CORE_NORMAL_INT_STAT_R_CMD_COMPLETE_Pos 0UL
#define SDHC_CORE_NORMAL_INT_STAT_R_CMD_COMPLETE_Msk 0x1UL
#define SDHC_CORE_NORMAL_INT_STAT_R_XFER_COMPLETE_Pos 1UL
#define SDHC_CORE_NORMAL_INT_STAT_R_XFER_COMPLETE_Msk 0x2UL
#define SDHC_CORE_NORMAL_INT_STAT_R_BGAP_EVENT_Pos 2UL
#define SDHC_CORE_NORMAL_INT_STAT_R_BGAP_EVENT_Msk 0x4UL
#define SDHC_CORE_NORMAL_INT_STAT_R_DMA_INTERRUPT_Pos 3UL
#define SDHC_CORE_NORMAL_INT_STAT_R_DMA_INTERRUPT_Msk 0x8UL
#define SDHC_CORE_NORMAL_INT_STAT_R_BUF_WR_READY_Pos 4UL
#define SDHC_CORE_NORMAL_INT_STAT_R_BUF_WR_READY_Msk 0x10UL
#define SDHC_CORE_NORMAL_INT_STAT_R_BUF_RD_READY_Pos 5UL
#define SDHC_CORE_NORMAL_INT_STAT_R_BUF_RD_READY_Msk 0x20UL
#define SDHC_CORE_NORMAL_INT_STAT_R_CARD_INSERTION_Pos 6UL
#define SDHC_CORE_NORMAL_INT_STAT_R_CARD_INSERTION_Msk 0x40UL
#define SDHC_CORE_NORMAL_INT_STAT_R_CARD_REMOVAL_Pos 7UL
#define SDHC_CORE_NORMAL_INT_STAT_R_CARD_REMOVAL_Msk 0x80UL
#define SDHC_CORE_NORMAL_INT_STAT_R_CARD_INTERRUPT_Pos 8UL
#define SDHC_CORE_NORMAL_INT_STAT_R_CARD_INTERRUPT_Msk 0x100UL
#define SDHC_CORE_NORMAL_INT_STAT_R_FX_EVENT_Pos 13UL
#define SDHC_CORE_NORMAL_INT_STAT_R_FX_EVENT_Msk 0x2000UL
#define SDHC_CORE_NORMAL_INT_STAT_R_CQE_EVENT_Pos 14UL
#define SDHC_CORE_NORMAL_INT_STAT_R_CQE_EVENT_Msk 0x4000UL
#define SDHC_CORE_NORMAL_INT_STAT_R_ERR_INTERRUPT_Pos 15UL
#define SDHC_CORE_NORMAL_INT_STAT_R_ERR_INTERRUPT_Msk 0x8000UL
/* SDHC_CORE.ERROR_INT_STAT_R */
#define SDHC_CORE_ERROR_INT_STAT_R_CMD_TOUT_ERR_Pos 0UL
#define SDHC_CORE_ERROR_INT_STAT_R_CMD_TOUT_ERR_Msk 0x1UL
#define SDHC_CORE_ERROR_INT_STAT_R_CMD_CRC_ERR_Pos 1UL
#define SDHC_CORE_ERROR_INT_STAT_R_CMD_CRC_ERR_Msk 0x2UL
#define SDHC_CORE_ERROR_INT_STAT_R_CMD_END_BIT_ERR_Pos 2UL
#define SDHC_CORE_ERROR_INT_STAT_R_CMD_END_BIT_ERR_Msk 0x4UL
#define SDHC_CORE_ERROR_INT_STAT_R_CMD_IDX_ERR_Pos 3UL
#define SDHC_CORE_ERROR_INT_STAT_R_CMD_IDX_ERR_Msk 0x8UL
#define SDHC_CORE_ERROR_INT_STAT_R_DATA_TOUT_ERR_Pos 4UL
#define SDHC_CORE_ERROR_INT_STAT_R_DATA_TOUT_ERR_Msk 0x10UL
#define SDHC_CORE_ERROR_INT_STAT_R_DATA_CRC_ERR_Pos 5UL
#define SDHC_CORE_ERROR_INT_STAT_R_DATA_CRC_ERR_Msk 0x20UL
#define SDHC_CORE_ERROR_INT_STAT_R_DATA_END_BIT_ERR_Pos 6UL
#define SDHC_CORE_ERROR_INT_STAT_R_DATA_END_BIT_ERR_Msk 0x40UL
#define SDHC_CORE_ERROR_INT_STAT_R_CUR_LMT_ERR_Pos 7UL
#define SDHC_CORE_ERROR_INT_STAT_R_CUR_LMT_ERR_Msk 0x80UL
#define SDHC_CORE_ERROR_INT_STAT_R_AUTO_CMD_ERR_Pos 8UL
#define SDHC_CORE_ERROR_INT_STAT_R_AUTO_CMD_ERR_Msk 0x100UL
#define SDHC_CORE_ERROR_INT_STAT_R_ADMA_ERR_Pos 9UL
#define SDHC_CORE_ERROR_INT_STAT_R_ADMA_ERR_Msk 0x200UL
#define SDHC_CORE_ERROR_INT_STAT_R_TUNING_ERR_Pos 10UL
#define SDHC_CORE_ERROR_INT_STAT_R_TUNING_ERR_Msk 0x400UL
#define SDHC_CORE_ERROR_INT_STAT_R_RESP_ERR_Pos 11UL
#define SDHC_CORE_ERROR_INT_STAT_R_RESP_ERR_Msk 0x800UL
#define SDHC_CORE_ERROR_INT_STAT_R_BOOT_ACK_ERR_Pos 12UL
#define SDHC_CORE_ERROR_INT_STAT_R_BOOT_ACK_ERR_Msk 0x1000UL
/* SDHC_CORE.NORMAL_INT_STAT_EN_R */
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CMD_COMPLETE_STAT_EN_Pos 0UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CMD_COMPLETE_STAT_EN_Msk 0x1UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_XFER_COMPLETE_STAT_EN_Pos 1UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_XFER_COMPLETE_STAT_EN_Msk 0x2UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_BGAP_EVENT_STAT_EN_Pos 2UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_BGAP_EVENT_STAT_EN_Msk 0x4UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_DMA_INTERRUPT_STAT_EN_Pos 3UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_DMA_INTERRUPT_STAT_EN_Msk 0x8UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_BUF_WR_READY_STAT_EN_Pos 4UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_BUF_WR_READY_STAT_EN_Msk 0x10UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_BUF_RD_READY_STAT_EN_Pos 5UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_BUF_RD_READY_STAT_EN_Msk 0x20UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CARD_INSERTION_STAT_EN_Pos 6UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CARD_INSERTION_STAT_EN_Msk 0x40UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CARD_REMOVAL_STAT_EN_Pos 7UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CARD_REMOVAL_STAT_EN_Msk 0x80UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CARD_INTERRUPT_STAT_EN_Pos 8UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CARD_INTERRUPT_STAT_EN_Msk 0x100UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_INT_A_STAT_EN_Pos 9UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_INT_A_STAT_EN_Msk 0x200UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_INT_B_STAT_EN_Pos 10UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_INT_B_STAT_EN_Msk 0x400UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_INT_C_STAT_EN_Pos 11UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_INT_C_STAT_EN_Msk 0x800UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_RE_TUNE_EVENT_STAT_EN_Pos 12UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_RE_TUNE_EVENT_STAT_EN_Msk 0x1000UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_FX_EVENT_STAT_EN_Pos 13UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_FX_EVENT_STAT_EN_Msk 0x2000UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CQE_EVENT_STAT_EN_Pos 14UL
#define SDHC_CORE_NORMAL_INT_STAT_EN_R_CQE_EVENT_STAT_EN_Msk 0x4000UL
/* SDHC_CORE.ERROR_INT_STAT_EN_R */
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CMD_TOUT_ERR_STAT_EN_Pos 0UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CMD_TOUT_ERR_STAT_EN_Msk 0x1UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CMD_CRC_ERR_STAT_EN_Pos 1UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CMD_CRC_ERR_STAT_EN_Msk 0x2UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CMD_END_BIT_ERR_STAT_EN_Pos 2UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CMD_END_BIT_ERR_STAT_EN_Msk 0x4UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CMD_IDX_ERR_STAT_EN_Pos 3UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CMD_IDX_ERR_STAT_EN_Msk 0x8UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_DATA_TOUT_ERR_STAT_EN_Pos 4UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_DATA_TOUT_ERR_STAT_EN_Msk 0x10UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_DATA_CRC_ERR_STAT_EN_Pos 5UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_DATA_CRC_ERR_STAT_EN_Msk 0x20UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_DATA_END_BIT_ERR_STAT_EN_Pos 6UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_DATA_END_BIT_ERR_STAT_EN_Msk 0x40UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CUR_LMT_ERR_STAT_EN_Pos 7UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_CUR_LMT_ERR_STAT_EN_Msk 0x80UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_AUTO_CMD_ERR_STAT_EN_Pos 8UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_AUTO_CMD_ERR_STAT_EN_Msk 0x100UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_ADMA_ERR_STAT_EN_Pos 9UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_ADMA_ERR_STAT_EN_Msk 0x200UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_TUNING_ERR_STAT_EN_Pos 10UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_TUNING_ERR_STAT_EN_Msk 0x400UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_RESP_ERR_STAT_EN_Pos 11UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_RESP_ERR_STAT_EN_Msk 0x800UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_BOOT_ACK_ERR_STAT_EN_Pos 12UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_BOOT_ACK_ERR_STAT_EN_Msk 0x1000UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_VENDOR_ERR_STAT_EN1_Pos 13UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_VENDOR_ERR_STAT_EN1_Msk 0x2000UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_VENDOR_ERR_STAT_EN2_Pos 14UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_VENDOR_ERR_STAT_EN2_Msk 0x4000UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_VENDOR_ERR_STAT_EN3_Pos 15UL
#define SDHC_CORE_ERROR_INT_STAT_EN_R_VENDOR_ERR_STAT_EN3_Msk 0x8000UL
/* SDHC_CORE.NORMAL_INT_SIGNAL_EN_R */
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CMD_COMPLETE_SIGNAL_EN_Pos 0UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CMD_COMPLETE_SIGNAL_EN_Msk 0x1UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_XFER_COMPLETE_SIGNAL_EN_Pos 1UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_XFER_COMPLETE_SIGNAL_EN_Msk 0x2UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_BGAP_EVENT_SIGNAL_EN_Pos 2UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_BGAP_EVENT_SIGNAL_EN_Msk 0x4UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_DMA_INTERRUPT_SIGNAL_EN_Pos 3UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_DMA_INTERRUPT_SIGNAL_EN_Msk 0x8UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_BUF_WR_READY_SIGNAL_EN_Pos 4UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_BUF_WR_READY_SIGNAL_EN_Msk 0x10UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_BUF_RD_READY_SIGNAL_EN_Pos 5UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_BUF_RD_READY_SIGNAL_EN_Msk 0x20UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CARD_INSERTION_SIGNAL_EN_Pos 6UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CARD_INSERTION_SIGNAL_EN_Msk 0x40UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CARD_REMOVAL_SIGNAL_EN_Pos 7UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CARD_REMOVAL_SIGNAL_EN_Msk 0x80UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CARD_INTERRUPT_SIGNAL_EN_Pos 8UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CARD_INTERRUPT_SIGNAL_EN_Msk 0x100UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_INT_A_SIGNAL_EN_Pos 9UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_INT_A_SIGNAL_EN_Msk 0x200UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_INT_B_SIGNAL_EN_Pos 10UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_INT_B_SIGNAL_EN_Msk 0x400UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_INT_C_SIGNAL_EN_Pos 11UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_INT_C_SIGNAL_EN_Msk 0x800UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_RE_TUNE_EVENT_SIGNAL_EN_Pos 12UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_RE_TUNE_EVENT_SIGNAL_EN_Msk 0x1000UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_FX_EVENT_SIGNAL_EN_Pos 13UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_FX_EVENT_SIGNAL_EN_Msk 0x2000UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CQE_EVENT_SIGNAL_EN_Pos 14UL
#define SDHC_CORE_NORMAL_INT_SIGNAL_EN_R_CQE_EVENT_SIGNAL_EN_Msk 0x4000UL
/* SDHC_CORE.ERROR_INT_SIGNAL_EN_R */
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CMD_TOUT_ERR_SIGNAL_EN_Pos 0UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CMD_TOUT_ERR_SIGNAL_EN_Msk 0x1UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CMD_CRC_ERR_SIGNAL_EN_Pos 1UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CMD_CRC_ERR_SIGNAL_EN_Msk 0x2UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CMD_END_BIT_ERR_SIGNAL_EN_Pos 2UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CMD_END_BIT_ERR_SIGNAL_EN_Msk 0x4UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CMD_IDX_ERR_SIGNAL_EN_Pos 3UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CMD_IDX_ERR_SIGNAL_EN_Msk 0x8UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_DATA_TOUT_ERR_SIGNAL_EN_Pos 4UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_DATA_TOUT_ERR_SIGNAL_EN_Msk 0x10UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_DATA_CRC_ERR_SIGNAL_EN_Pos 5UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_DATA_CRC_ERR_SIGNAL_EN_Msk 0x20UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_DATA_END_BIT_ERR_SIGNAL_EN_Pos 6UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_DATA_END_BIT_ERR_SIGNAL_EN_Msk 0x40UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CUR_LMT_ERR_SIGNAL_EN_Pos 7UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_CUR_LMT_ERR_SIGNAL_EN_Msk 0x80UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_AUTO_CMD_ERR_SIGNAL_EN_Pos 8UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_AUTO_CMD_ERR_SIGNAL_EN_Msk 0x100UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_ADMA_ERR_SIGNAL_EN_Pos 9UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_ADMA_ERR_SIGNAL_EN_Msk 0x200UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_TUNING_ERR_SIGNAL_EN_Pos 10UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_TUNING_ERR_SIGNAL_EN_Msk 0x400UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_RESP_ERR_SIGNAL_EN_Pos 11UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_RESP_ERR_SIGNAL_EN_Msk 0x800UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_BOOT_ACK_ERR_SIGNAL_EN_Pos 12UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_BOOT_ACK_ERR_SIGNAL_EN_Msk 0x1000UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_VENDOR_ERR_SIGNAL_EN1_Pos 13UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_VENDOR_ERR_SIGNAL_EN1_Msk 0x2000UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_VENDOR_ERR_SIGNAL_EN2_Pos 14UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_VENDOR_ERR_SIGNAL_EN2_Msk 0x4000UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_VENDOR_ERR_SIGNAL_EN3_Pos 15UL
#define SDHC_CORE_ERROR_INT_SIGNAL_EN_R_VENDOR_ERR_SIGNAL_EN3_Msk 0x8000UL
/* SDHC_CORE.AUTO_CMD_STAT_R */
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD12_NOT_EXEC_Pos 0UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD12_NOT_EXEC_Msk 0x1UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_TOUT_ERR_Pos 1UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_TOUT_ERR_Msk 0x2UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_CRC_ERR_Pos 2UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_CRC_ERR_Msk 0x4UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_EBIT_ERR_Pos 3UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_EBIT_ERR_Msk 0x8UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_IDX_ERR_Pos 4UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_IDX_ERR_Msk 0x10UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_RESP_ERR_Pos 5UL
#define SDHC_CORE_AUTO_CMD_STAT_R_AUTO_CMD_RESP_ERR_Msk 0x20UL
#define SDHC_CORE_AUTO_CMD_STAT_R_CMD_NOT_ISSUED_AUTO_CMD12_Pos 7UL
#define SDHC_CORE_AUTO_CMD_STAT_R_CMD_NOT_ISSUED_AUTO_CMD12_Msk 0x80UL
/* SDHC_CORE.HOST_CTRL2_R */
#define SDHC_CORE_HOST_CTRL2_R_UHS_MODE_SEL_Pos 0UL
#define SDHC_CORE_HOST_CTRL2_R_UHS_MODE_SEL_Msk 0x7UL
#define SDHC_CORE_HOST_CTRL2_R_SIGNALING_EN_Pos 3UL
#define SDHC_CORE_HOST_CTRL2_R_SIGNALING_EN_Msk 0x8UL
#define SDHC_CORE_HOST_CTRL2_R_DRV_STRENGTH_SEL_Pos 4UL
#define SDHC_CORE_HOST_CTRL2_R_DRV_STRENGTH_SEL_Msk 0x30UL
#define SDHC_CORE_HOST_CTRL2_R_EXEC_TUNING_Pos  6UL
#define SDHC_CORE_HOST_CTRL2_R_EXEC_TUNING_Msk  0x40UL
#define SDHC_CORE_HOST_CTRL2_R_SAMPLE_CLK_SEL_Pos 7UL
#define SDHC_CORE_HOST_CTRL2_R_SAMPLE_CLK_SEL_Msk 0x80UL
#define SDHC_CORE_HOST_CTRL2_R_UHS2_IF_ENABLE_Pos 8UL
#define SDHC_CORE_HOST_CTRL2_R_UHS2_IF_ENABLE_Msk 0x100UL
#define SDHC_CORE_HOST_CTRL2_R_ADMA2_LEN_MODE_Pos 10UL
#define SDHC_CORE_HOST_CTRL2_R_ADMA2_LEN_MODE_Msk 0x400UL
#define SDHC_CORE_HOST_CTRL2_R_CMD23_ENABLE_Pos 11UL
#define SDHC_CORE_HOST_CTRL2_R_CMD23_ENABLE_Msk 0x800UL
#define SDHC_CORE_HOST_CTRL2_R_HOST_VER4_ENABLE_Pos 12UL
#define SDHC_CORE_HOST_CTRL2_R_HOST_VER4_ENABLE_Msk 0x1000UL
#define SDHC_CORE_HOST_CTRL2_R_ADDRESSING_Pos   13UL
#define SDHC_CORE_HOST_CTRL2_R_ADDRESSING_Msk   0x2000UL
#define SDHC_CORE_HOST_CTRL2_R_ASYNC_INT_ENABLE_Pos 14UL
#define SDHC_CORE_HOST_CTRL2_R_ASYNC_INT_ENABLE_Msk 0x4000UL
#define SDHC_CORE_HOST_CTRL2_R_PRESET_VAL_ENABLE_Pos 15UL
#define SDHC_CORE_HOST_CTRL2_R_PRESET_VAL_ENABLE_Msk 0x8000UL
/* SDHC_CORE.CAPABILITIES1_R */
#define SDHC_CORE_CAPABILITIES1_R_TOUT_CLK_FREQ_Pos 0UL
#define SDHC_CORE_CAPABILITIES1_R_TOUT_CLK_FREQ_Msk 0x3FUL
#define SDHC_CORE_CAPABILITIES1_R_TOUT_CLK_UNIT_Pos 7UL
#define SDHC_CORE_CAPABILITIES1_R_TOUT_CLK_UNIT_Msk 0x80UL
#define SDHC_CORE_CAPABILITIES1_R_BASE_CLK_FREQ_Pos 8UL
#define SDHC_CORE_CAPABILITIES1_R_BASE_CLK_FREQ_Msk 0xFF00UL
#define SDHC_CORE_CAPABILITIES1_R_MAX_BLK_LEN_Pos 16UL
#define SDHC_CORE_CAPABILITIES1_R_MAX_BLK_LEN_Msk 0x30000UL
#define SDHC_CORE_CAPABILITIES1_R_EMBEDDED_8_BIT_Pos 18UL
#define SDHC_CORE_CAPABILITIES1_R_EMBEDDED_8_BIT_Msk 0x40000UL
#define SDHC_CORE_CAPABILITIES1_R_ADMA2_SUPPORT_Pos 19UL
#define SDHC_CORE_CAPABILITIES1_R_ADMA2_SUPPORT_Msk 0x80000UL
#define SDHC_CORE_CAPABILITIES1_R_HIGH_SPEED_SUPPORT_Pos 21UL
#define SDHC_CORE_CAPABILITIES1_R_HIGH_SPEED_SUPPORT_Msk 0x200000UL
#define SDHC_CORE_CAPABILITIES1_R_SDMA_SUPPORT_Pos 22UL
#define SDHC_CORE_CAPABILITIES1_R_SDMA_SUPPORT_Msk 0x400000UL
#define SDHC_CORE_CAPABILITIES1_R_SUS_RES_SUPPORT_Pos 23UL
#define SDHC_CORE_CAPABILITIES1_R_SUS_RES_SUPPORT_Msk 0x800000UL
#define SDHC_CORE_CAPABILITIES1_R_VOLT_33_Pos   24UL
#define SDHC_CORE_CAPABILITIES1_R_VOLT_33_Msk   0x1000000UL
#define SDHC_CORE_CAPABILITIES1_R_VOLT_30_Pos   25UL
#define SDHC_CORE_CAPABILITIES1_R_VOLT_30_Msk   0x2000000UL
#define SDHC_CORE_CAPABILITIES1_R_VOLT_18_Pos   26UL
#define SDHC_CORE_CAPABILITIES1_R_VOLT_18_Msk   0x4000000UL
#define SDHC_CORE_CAPABILITIES1_R_SYS_ADDR_64_V4_Pos 27UL
#define SDHC_CORE_CAPABILITIES1_R_SYS_ADDR_64_V4_Msk 0x8000000UL
#define SDHC_CORE_CAPABILITIES1_R_SYS_ADDR_64_V3_Pos 28UL
#define SDHC_CORE_CAPABILITIES1_R_SYS_ADDR_64_V3_Msk 0x10000000UL
#define SDHC_CORE_CAPABILITIES1_R_ASYNC_INT_SUPPORT_Pos 29UL
#define SDHC_CORE_CAPABILITIES1_R_ASYNC_INT_SUPPORT_Msk 0x20000000UL
#define SDHC_CORE_CAPABILITIES1_R_SLOT_TYPE_R_Pos 30UL
#define SDHC_CORE_CAPABILITIES1_R_SLOT_TYPE_R_Msk 0xC0000000UL
/* SDHC_CORE.CAPABILITIES2_R */
#define SDHC_CORE_CAPABILITIES2_R_SDR50_SUPPORT_Pos 0UL
#define SDHC_CORE_CAPABILITIES2_R_SDR50_SUPPORT_Msk 0x1UL
#define SDHC_CORE_CAPABILITIES2_R_SDR104_SUPPORT_Pos 1UL
#define SDHC_CORE_CAPABILITIES2_R_SDR104_SUPPORT_Msk 0x2UL
#define SDHC_CORE_CAPABILITIES2_R_DDR50_SUPPORT_Pos 2UL
#define SDHC_CORE_CAPABILITIES2_R_DDR50_SUPPORT_Msk 0x4UL
#define SDHC_CORE_CAPABILITIES2_R_UHS2_SUPPORT_Pos 3UL
#define SDHC_CORE_CAPABILITIES2_R_UHS2_SUPPORT_Msk 0x8UL
#define SDHC_CORE_CAPABILITIES2_R_DRV_TYPEA_Pos 4UL
#define SDHC_CORE_CAPABILITIES2_R_DRV_TYPEA_Msk 0x10UL
#define SDHC_CORE_CAPABILITIES2_R_DRV_TYPEC_Pos 5UL
#define SDHC_CORE_CAPABILITIES2_R_DRV_TYPEC_Msk 0x20UL
#define SDHC_CORE_CAPABILITIES2_R_DRV_TYPED_Pos 6UL
#define SDHC_CORE_CAPABILITIES2_R_DRV_TYPED_Msk 0x40UL
#define SDHC_CORE_CAPABILITIES2_R_RETUNE_CNT_Pos 8UL
#define SDHC_CORE_CAPABILITIES2_R_RETUNE_CNT_Msk 0xF00UL
#define SDHC_CORE_CAPABILITIES2_R_USE_TUNING_SDR50_Pos 13UL
#define SDHC_CORE_CAPABILITIES2_R_USE_TUNING_SDR50_Msk 0x2000UL
#define SDHC_CORE_CAPABILITIES2_R_RE_TUNING_MODES_Pos 14UL
#define SDHC_CORE_CAPABILITIES2_R_RE_TUNING_MODES_Msk 0xC000UL
#define SDHC_CORE_CAPABILITIES2_R_CLK_MUL_Pos   16UL
#define SDHC_CORE_CAPABILITIES2_R_CLK_MUL_Msk   0xFF0000UL
#define SDHC_CORE_CAPABILITIES2_R_ADMA3_SUPPORT_Pos 27UL
#define SDHC_CORE_CAPABILITIES2_R_ADMA3_SUPPORT_Msk 0x8000000UL
#define SDHC_CORE_CAPABILITIES2_R_VDD2_18V_SUPPORT_Pos 28UL
#define SDHC_CORE_CAPABILITIES2_R_VDD2_18V_SUPPORT_Msk 0x10000000UL
/* SDHC_CORE.CURR_CAPABILITIES1_R */
#define SDHC_CORE_CURR_CAPABILITIES1_R_MAX_CUR_33V_Pos 0UL
#define SDHC_CORE_CURR_CAPABILITIES1_R_MAX_CUR_33V_Msk 0xFFUL
#define SDHC_CORE_CURR_CAPABILITIES1_R_MAX_CUR_30V_Pos 8UL
#define SDHC_CORE_CURR_CAPABILITIES1_R_MAX_CUR_30V_Msk 0xFF00UL
#define SDHC_CORE_CURR_CAPABILITIES1_R_MAX_CUR_18V_Pos 16UL
#define SDHC_CORE_CURR_CAPABILITIES1_R_MAX_CUR_18V_Msk 0xFF0000UL
/* SDHC_CORE.CURR_CAPABILITIES2_R */
#define SDHC_CORE_CURR_CAPABILITIES2_R_MAX_CUR_VDD2_18V_Pos 0UL
#define SDHC_CORE_CURR_CAPABILITIES2_R_MAX_CUR_VDD2_18V_Msk 0xFFUL
/* SDHC_CORE.FORCE_AUTO_CMD_STAT_R */
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD12_NOT_EXEC_Pos 0UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD12_NOT_EXEC_Msk 0x1UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_TOUT_ERR_Pos 1UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_TOUT_ERR_Msk 0x2UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_CRC_ERR_Pos 2UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_CRC_ERR_Msk 0x4UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_EBIT_ERR_Pos 3UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_EBIT_ERR_Msk 0x8UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_IDX_ERR_Pos 4UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_IDX_ERR_Msk 0x10UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_RESP_ERR_Pos 5UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_AUTO_CMD_RESP_ERR_Msk 0x20UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_CMD_NOT_ISSUED_AUTO_CMD12_Pos 7UL
#define SDHC_CORE_FORCE_AUTO_CMD_STAT_R_FORCE_CMD_NOT_ISSUED_AUTO_CMD12_Msk 0x80UL
/* SDHC_CORE.FORCE_ERROR_INT_STAT_R */
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CMD_TOUT_ERR_Pos 0UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CMD_TOUT_ERR_Msk 0x1UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CMD_CRC_ERR_Pos 1UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CMD_CRC_ERR_Msk 0x2UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CMD_END_BIT_ERR_Pos 2UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CMD_END_BIT_ERR_Msk 0x4UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CMD_IDX_ERR_Pos 3UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CMD_IDX_ERR_Msk 0x8UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_DATA_TOUT_ERR_Pos 4UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_DATA_TOUT_ERR_Msk 0x10UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_DATA_CRC_ERR_Pos 5UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_DATA_CRC_ERR_Msk 0x20UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_DATA_END_BIT_ERR_Pos 6UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_DATA_END_BIT_ERR_Msk 0x40UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CUR_LMT_ERR_Pos 7UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_CUR_LMT_ERR_Msk 0x80UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_AUTO_CMD_ERR_Pos 8UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_AUTO_CMD_ERR_Msk 0x100UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_ADMA_ERR_Pos 9UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_ADMA_ERR_Msk 0x200UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_TUNING_ERR_Pos 10UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_TUNING_ERR_Msk 0x400UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_RESP_ERR_Pos 11UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_RESP_ERR_Msk 0x800UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_BOOT_ACK_ERR_Pos 12UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_BOOT_ACK_ERR_Msk 0x1000UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_VENDOR_ERR1_Pos 13UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_VENDOR_ERR1_Msk 0x2000UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_VENDOR_ERR2_Pos 14UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_VENDOR_ERR2_Msk 0x4000UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_VENDOR_ERR3_Pos 15UL
#define SDHC_CORE_FORCE_ERROR_INT_STAT_R_FORCE_VENDOR_ERR3_Msk 0x8000UL
/* SDHC_CORE.ADMA_ERR_STAT_R */
#define SDHC_CORE_ADMA_ERR_STAT_R_ADMA_ERR_STATES_Pos 0UL
#define SDHC_CORE_ADMA_ERR_STAT_R_ADMA_ERR_STATES_Msk 0x3UL
#define SDHC_CORE_ADMA_ERR_STAT_R_ADMA_LEN_ERR_Pos 2UL
#define SDHC_CORE_ADMA_ERR_STAT_R_ADMA_LEN_ERR_Msk 0x4UL
/* SDHC_CORE.ADMA_SA_LOW_R */
#define SDHC_CORE_ADMA_SA_LOW_R_ADMA_SA_LOW_Pos 0UL
#define SDHC_CORE_ADMA_SA_LOW_R_ADMA_SA_LOW_Msk 0xFFFFFFFFUL
/* SDHC_CORE.ADMA_ID_LOW_R */
#define SDHC_CORE_ADMA_ID_LOW_R_ADMA_ID_LOW_Pos 0UL
#define SDHC_CORE_ADMA_ID_LOW_R_ADMA_ID_LOW_Msk 0xFFFFFFFFUL
/* SDHC_CORE.HOST_CNTRL_VERS_R */
#define SDHC_CORE_HOST_CNTRL_VERS_R_SPEC_VERSION_NUM_Pos 0UL
#define SDHC_CORE_HOST_CNTRL_VERS_R_SPEC_VERSION_NUM_Msk 0xFFUL
#define SDHC_CORE_HOST_CNTRL_VERS_R_VENDOR_VERSION_NUM_Pos 8UL
#define SDHC_CORE_HOST_CNTRL_VERS_R_VENDOR_VERSION_NUM_Msk 0xFF00UL
/* SDHC_CORE.CQVER */
#define SDHC_CORE_CQVER_EMMC_VER_SUFFIX_Pos     0UL
#define SDHC_CORE_CQVER_EMMC_VER_SUFFIX_Msk     0xFUL
#define SDHC_CORE_CQVER_EMMC_VER_MINOR_Pos      4UL
#define SDHC_CORE_CQVER_EMMC_VER_MINOR_Msk      0xF0UL
#define SDHC_CORE_CQVER_EMMC_VER_MAJOR_Pos      8UL
#define SDHC_CORE_CQVER_EMMC_VER_MAJOR_Msk      0xF00UL
/* SDHC_CORE.CQCAP */
#define SDHC_CORE_CQCAP_ITCFVAL_Pos             0UL
#define SDHC_CORE_CQCAP_ITCFVAL_Msk             0x3FFUL
#define SDHC_CORE_CQCAP_ITCFMUL_Pos             12UL
#define SDHC_CORE_CQCAP_ITCFMUL_Msk             0xF000UL
#define SDHC_CORE_CQCAP_CRYPTO_SUPPORT_Pos      28UL
#define SDHC_CORE_CQCAP_CRYPTO_SUPPORT_Msk      0x10000000UL
/* SDHC_CORE.CQCFG */
#define SDHC_CORE_CQCFG_CQ_EN_Pos               0UL
#define SDHC_CORE_CQCFG_CQ_EN_Msk               0x1UL
#define SDHC_CORE_CQCFG_CR_GENERAL_EN_Pos       1UL
#define SDHC_CORE_CQCFG_CR_GENERAL_EN_Msk       0x2UL
#define SDHC_CORE_CQCFG_TASK_DESC_SIZE_Pos      8UL
#define SDHC_CORE_CQCFG_TASK_DESC_SIZE_Msk      0x100UL
#define SDHC_CORE_CQCFG_DCMD_EN_Pos             12UL
#define SDHC_CORE_CQCFG_DCMD_EN_Msk             0x1000UL
/* SDHC_CORE.CQCTL */
#define SDHC_CORE_CQCTL_HALT_Pos                0UL
#define SDHC_CORE_CQCTL_HALT_Msk                0x1UL
#define SDHC_CORE_CQCTL_CLR_ALL_TASKS_Pos       8UL
#define SDHC_CORE_CQCTL_CLR_ALL_TASKS_Msk       0x100UL
/* SDHC_CORE.CQIS */
#define SDHC_CORE_CQIS_HAC_Pos                  0UL
#define SDHC_CORE_CQIS_HAC_Msk                  0x1UL
#define SDHC_CORE_CQIS_TCC_Pos                  1UL
#define SDHC_CORE_CQIS_TCC_Msk                  0x2UL
#define SDHC_CORE_CQIS_RED_Pos                  2UL
#define SDHC_CORE_CQIS_RED_Msk                  0x4UL
#define SDHC_CORE_CQIS_TCL_Pos                  3UL
#define SDHC_CORE_CQIS_TCL_Msk                  0x8UL
#define SDHC_CORE_CQIS_GCE_Pos                  4UL
#define SDHC_CORE_CQIS_GCE_Msk                  0x10UL
#define SDHC_CORE_CQIS_ICCE_Pos                 5UL
#define SDHC_CORE_CQIS_ICCE_Msk                 0x20UL
/* SDHC_CORE.CQISE */
#define SDHC_CORE_CQISE_HAC_STE_Pos             0UL
#define SDHC_CORE_CQISE_HAC_STE_Msk             0x1UL
#define SDHC_CORE_CQISE_TCC_STE_Pos             1UL
#define SDHC_CORE_CQISE_TCC_STE_Msk             0x2UL
#define SDHC_CORE_CQISE_RED_STE_Pos             2UL
#define SDHC_CORE_CQISE_RED_STE_Msk             0x4UL
#define SDHC_CORE_CQISE_TCL_STE_Pos             3UL
#define SDHC_CORE_CQISE_TCL_STE_Msk             0x8UL
#define SDHC_CORE_CQISE_GCE_STE_Pos             4UL
#define SDHC_CORE_CQISE_GCE_STE_Msk             0x10UL
#define SDHC_CORE_CQISE_ICCE_STE_Pos            5UL
#define SDHC_CORE_CQISE_ICCE_STE_Msk            0x20UL
/* SDHC_CORE.CQISGE */
#define SDHC_CORE_CQISGE_HAC_SGE_Pos            0UL
#define SDHC_CORE_CQISGE_HAC_SGE_Msk            0x1UL
#define SDHC_CORE_CQISGE_TCC_SGE_Pos            1UL
#define SDHC_CORE_CQISGE_TCC_SGE_Msk            0x2UL
#define SDHC_CORE_CQISGE_RED_SGE_Pos            2UL
#define SDHC_CORE_CQISGE_RED_SGE_Msk            0x4UL
#define SDHC_CORE_CQISGE_TCL_SGE_Pos            3UL
#define SDHC_CORE_CQISGE_TCL_SGE_Msk            0x8UL
#define SDHC_CORE_CQISGE_GCE_SGE_Pos            4UL
#define SDHC_CORE_CQISGE_GCE_SGE_Msk            0x10UL
#define SDHC_CORE_CQISGE_ICCE_SGE_Pos           5UL
#define SDHC_CORE_CQISGE_ICCE_SGE_Msk           0x20UL
/* SDHC_CORE.CQIC */
#define SDHC_CORE_CQIC_TOUT_VAL_Pos             0UL
#define SDHC_CORE_CQIC_TOUT_VAL_Msk             0x7FUL
#define SDHC_CORE_CQIC_TOUT_VAL_WEN_Pos         7UL
#define SDHC_CORE_CQIC_TOUT_VAL_WEN_Msk         0x80UL
#define SDHC_CORE_CQIC_INTC_TH_Pos              8UL
#define SDHC_CORE_CQIC_INTC_TH_Msk              0x1F00UL
#define SDHC_CORE_CQIC_INTC_TH_WEN_Pos          15UL
#define SDHC_CORE_CQIC_INTC_TH_WEN_Msk          0x8000UL
#define SDHC_CORE_CQIC_INTC_RST_Pos             16UL
#define SDHC_CORE_CQIC_INTC_RST_Msk             0x10000UL
#define SDHC_CORE_CQIC_INTC_STAT_Pos            20UL
#define SDHC_CORE_CQIC_INTC_STAT_Msk            0x100000UL
#define SDHC_CORE_CQIC_INTC_EN_Pos              31UL
#define SDHC_CORE_CQIC_INTC_EN_Msk              0x80000000UL
/* SDHC_CORE.CQTDLBA */
#define SDHC_CORE_CQTDLBA_TDLBA_Pos             0UL
#define SDHC_CORE_CQTDLBA_TDLBA_Msk             0xFFFFFFFFUL
/* SDHC_CORE.CQTDBR */
#define SDHC_CORE_CQTDBR_DBR_Pos                0UL
#define SDHC_CORE_CQTDBR_DBR_Msk                0xFFFFFFFFUL
/* SDHC_CORE.CQTCN */
#define SDHC_CORE_CQTCN_TCN_Pos                 0UL
#define SDHC_CORE_CQTCN_TCN_Msk                 0xFFFFFFFFUL
/* SDHC_CORE.CQDQS */
#define SDHC_CORE_CQDQS_DQS_Pos                 0UL
#define SDHC_CORE_CQDQS_DQS_Msk                 0xFFFFFFFFUL
/* SDHC_CORE.CQDPT */
#define SDHC_CORE_CQDPT_DPT_Pos                 0UL
#define SDHC_CORE_CQDPT_DPT_Msk                 0xFFFFFFFFUL
/* SDHC_CORE.CQTCLR */
#define SDHC_CORE_CQTCLR_TCLR_Pos               0UL
#define SDHC_CORE_CQTCLR_TCLR_Msk               0xFFFFFFFFUL
/* SDHC_CORE.CQSSC1 */
#define SDHC_CORE_CQSSC1_SQSCMD_IDLE_TMR_Pos    0UL
#define SDHC_CORE_CQSSC1_SQSCMD_IDLE_TMR_Msk    0xFFFFUL
#define SDHC_CORE_CQSSC1_SQSCMD_BLK_CNT_Pos     16UL
#define SDHC_CORE_CQSSC1_SQSCMD_BLK_CNT_Msk     0xF0000UL
/* SDHC_CORE.CQSSC2 */
#define SDHC_CORE_CQSSC2_SQSCMD_RCA_Pos         0UL
#define SDHC_CORE_CQSSC2_SQSCMD_RCA_Msk         0xFFFFUL
/* SDHC_CORE.CQCRDCT */
#define SDHC_CORE_CQCRDCT_DCMD_RESP_Pos         0UL
#define SDHC_CORE_CQCRDCT_DCMD_RESP_Msk         0xFFFFFFFFUL
/* SDHC_CORE.CQRMEM */
#define SDHC_CORE_CQRMEM_RESP_ERR_MASK_Pos      0UL
#define SDHC_CORE_CQRMEM_RESP_ERR_MASK_Msk      0xFFFFFFFFUL
/* SDHC_CORE.CQTERRI */
#define SDHC_CORE_CQTERRI_RESP_ERR_CMD_INDX_Pos 0UL
#define SDHC_CORE_CQTERRI_RESP_ERR_CMD_INDX_Msk 0x3FUL
#define SDHC_CORE_CQTERRI_RESP_ERR_TASKID_Pos   8UL
#define SDHC_CORE_CQTERRI_RESP_ERR_TASKID_Msk   0x1F00UL
#define SDHC_CORE_CQTERRI_RESP_ERR_FIELDS_VALID_Pos 15UL
#define SDHC_CORE_CQTERRI_RESP_ERR_FIELDS_VALID_Msk 0x8000UL
#define SDHC_CORE_CQTERRI_TRANS_ERR_CMD_INDX_Pos 16UL
#define SDHC_CORE_CQTERRI_TRANS_ERR_CMD_INDX_Msk 0x3F0000UL
#define SDHC_CORE_CQTERRI_TRANS_ERR_TASKID_Pos  24UL
#define SDHC_CORE_CQTERRI_TRANS_ERR_TASKID_Msk  0x1F000000UL
#define SDHC_CORE_CQTERRI_TRANS_ERR_FIELDS_VALID_Pos 31UL
#define SDHC_CORE_CQTERRI_TRANS_ERR_FIELDS_VALID_Msk 0x80000000UL
/* SDHC_CORE.CQCRI */
#define SDHC_CORE_CQCRI_CMD_RESP_INDX_Pos       0UL
#define SDHC_CORE_CQCRI_CMD_RESP_INDX_Msk       0x3FUL
/* SDHC_CORE.CQCRA */
#define SDHC_CORE_CQCRA_CMD_RESP_ARG_Pos        0UL
#define SDHC_CORE_CQCRA_CMD_RESP_ARG_Msk        0xFFFFFFFFUL
/* SDHC_CORE.MSHC_VER_ID_R */
#define SDHC_CORE_MSHC_VER_ID_R_MSHC_VER_ID_Pos 0UL
#define SDHC_CORE_MSHC_VER_ID_R_MSHC_VER_ID_Msk 0xFFFFFFFFUL
/* SDHC_CORE.MSHC_VER_TYPE_R */
#define SDHC_CORE_MSHC_VER_TYPE_R_MSHC_VER_TYPE_Pos 0UL
#define SDHC_CORE_MSHC_VER_TYPE_R_MSHC_VER_TYPE_Msk 0xFFFFFFFFUL
/* SDHC_CORE.MSHC_CTRL_R */
#define SDHC_CORE_MSHC_CTRL_R_CMD_CONFLICT_CHECK_Pos 0UL
#define SDHC_CORE_MSHC_CTRL_R_CMD_CONFLICT_CHECK_Msk 0x1UL
#define SDHC_CORE_MSHC_CTRL_R_SW_CG_DIS_Pos     4UL
#define SDHC_CORE_MSHC_CTRL_R_SW_CG_DIS_Msk     0x10UL
/* SDHC_CORE.MBIU_CTRL_R */
#define SDHC_CORE_MBIU_CTRL_R_UNDEFL_INCR_EN_Pos 0UL
#define SDHC_CORE_MBIU_CTRL_R_UNDEFL_INCR_EN_Msk 0x1UL
#define SDHC_CORE_MBIU_CTRL_R_BURST_INCR4_EN_Pos 1UL
#define SDHC_CORE_MBIU_CTRL_R_BURST_INCR4_EN_Msk 0x2UL
#define SDHC_CORE_MBIU_CTRL_R_BURST_INCR8_EN_Pos 2UL
#define SDHC_CORE_MBIU_CTRL_R_BURST_INCR8_EN_Msk 0x4UL
#define SDHC_CORE_MBIU_CTRL_R_BURST_INCR16_EN_Pos 3UL
#define SDHC_CORE_MBIU_CTRL_R_BURST_INCR16_EN_Msk 0x8UL
/* SDHC_CORE.EMMC_CTRL_R */
#define SDHC_CORE_EMMC_CTRL_R_CARD_IS_EMMC_Pos  0UL
#define SDHC_CORE_EMMC_CTRL_R_CARD_IS_EMMC_Msk  0x1UL
#define SDHC_CORE_EMMC_CTRL_R_DISABLE_DATA_CRC_CHK_Pos 1UL
#define SDHC_CORE_EMMC_CTRL_R_DISABLE_DATA_CRC_CHK_Msk 0x2UL
#define SDHC_CORE_EMMC_CTRL_R_EMMC_RST_N_Pos    2UL
#define SDHC_CORE_EMMC_CTRL_R_EMMC_RST_N_Msk    0x4UL
#define SDHC_CORE_EMMC_CTRL_R_EMMC_RST_N_OE_Pos 3UL
#define SDHC_CORE_EMMC_CTRL_R_EMMC_RST_N_OE_Msk 0x8UL
#define SDHC_CORE_EMMC_CTRL_R_CQE_ALGO_SEL_Pos  9UL
#define SDHC_CORE_EMMC_CTRL_R_CQE_ALGO_SEL_Msk  0x200UL
#define SDHC_CORE_EMMC_CTRL_R_CQE_PREFETCH_DISABLE_Pos 10UL
#define SDHC_CORE_EMMC_CTRL_R_CQE_PREFETCH_DISABLE_Msk 0x400UL
/* SDHC_CORE.BOOT_CTRL_R */
#define SDHC_CORE_BOOT_CTRL_R_MAN_BOOT_EN_Pos   0UL
#define SDHC_CORE_BOOT_CTRL_R_MAN_BOOT_EN_Msk   0x1UL
#define SDHC_CORE_BOOT_CTRL_R_VALIDATE_BOOT_Pos 7UL
#define SDHC_CORE_BOOT_CTRL_R_VALIDATE_BOOT_Msk 0x80UL
#define SDHC_CORE_BOOT_CTRL_R_BOOT_ACK_ENABLE_Pos 8UL
#define SDHC_CORE_BOOT_CTRL_R_BOOT_ACK_ENABLE_Msk 0x100UL
#define SDHC_CORE_BOOT_CTRL_R_BOOT_TOUT_CNT_Pos 12UL
#define SDHC_CORE_BOOT_CTRL_R_BOOT_TOUT_CNT_Msk 0xF000UL
/* SDHC_CORE.GP_IN_R */
#define SDHC_CORE_GP_IN_R_GP_IN_Pos             0UL
#define SDHC_CORE_GP_IN_R_GP_IN_Msk             0x1UL
/* SDHC_CORE.GP_OUT_R */
#define SDHC_CORE_GP_OUT_R_CARD_DETECT_EN_Pos   0UL
#define SDHC_CORE_GP_OUT_R_CARD_DETECT_EN_Msk   0x1UL
#define SDHC_CORE_GP_OUT_R_CARD_MECH_WRITE_PROT_EN_Pos 1UL
#define SDHC_CORE_GP_OUT_R_CARD_MECH_WRITE_PROT_EN_Msk 0x2UL
#define SDHC_CORE_GP_OUT_R_LED_CTRL_OE_Pos      2UL
#define SDHC_CORE_GP_OUT_R_LED_CTRL_OE_Msk      0x4UL
#define SDHC_CORE_GP_OUT_R_CARD_CLOCK_OE_Pos    3UL
#define SDHC_CORE_GP_OUT_R_CARD_CLOCK_OE_Msk    0x8UL
#define SDHC_CORE_GP_OUT_R_CARD_IF_PWR_EN_OE_Pos 4UL
#define SDHC_CORE_GP_OUT_R_CARD_IF_PWR_EN_OE_Msk 0x10UL
#define SDHC_CORE_GP_OUT_R_IO_VOLT_SEL_OE_Pos   5UL
#define SDHC_CORE_GP_OUT_R_IO_VOLT_SEL_OE_Msk   0x20UL
#define SDHC_CORE_GP_OUT_R_CARD_CLOCK_OUT_DLY_Pos 6UL
#define SDHC_CORE_GP_OUT_R_CARD_CLOCK_OUT_DLY_Msk 0xC0UL
#define SDHC_CORE_GP_OUT_R_CARD_CLOCK_IN_DLY_Pos 8UL
#define SDHC_CORE_GP_OUT_R_CARD_CLOCK_IN_DLY_Msk 0x300UL


#endif /* _CYIP_SDHC_H_ */


/* [] END OF FILE */
