"""
作者： qingchen
日期： 2024/1/12
"""
import datetime
import json

import redis
import redis_lock

from conf import setting

# redis 操作


pool = redis.ConnectionPool(host=setting.redis_conf['redis_host'], password=setting.redis_conf['redis_password'],
                            port=setting.redis_conf['redis_port'], db=setting.redis_conf['redis_db'],
                            decode_responses=True)


class RedisWithLock:
    def __init__(self):
        self.redis_cli = redis.Redis(connection_pool=pool)
        # self.time_ = datetime.datetime.now().strftime('%Y%m%d')

    def __del__(self):
        if self.redis_cli:
            self.redis_cli.close()


    def __enter__(self):
        # 在进入with块时调用，返回资源对象
        # self.redis_cli = redis.Redis(connection_pool=pool)
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        # 在离开with块时调用，用于资源释放
        self.close()

    def close(self):
        self.redis_cli.close()

    def redis_hash_is_exist_key(self, _hash_name: str, key: str):
        """  哈希
        判断键值是否存在
        :param key:  健值
        :return:
        """
        lock = redis_lock.Lock(self.redis_cli, str(_hash_name), expire=15)
        with lock:
            if self.redis_cli.hexists(_hash_name, key):
                return True
            else:
                return False

    def redis_update(self, _hash_name: str, key: str, value):
        """   哈希
         更新键值  上锁
        :param key:  健值
        :param value:  需要更新的内容  字典
        :return:
        """
        if isinstance(value, dict):
            value = json.dumps(value)
        lock = redis_lock.Lock(self.redis_cli, str(_hash_name), expire=15)
        with lock:
            if self.redis_cli.hset(_hash_name, key, value):
                # 存在该键值
                pass
            else:
                # 键不存在  需要创建该键
                self.redis_cli.hset(_hash_name, key, value)

    def redis_get(self, _hash_name: str, key: str):
        """  哈希
          # 获取健的值  并上锁
        :param key:  键的名称
        :return: 值  类型 dict
        """
        lock = redis_lock.Lock(self.redis_cli, str(_hash_name), expire=15)
        with lock:
            value = self.redis_cli.hget(_hash_name, key)
        if value:
            try:
                return json.loads(value)
            except:
                return value
        else:
            return None

    def record_status(self, key, value):
        """
        记录状态  list   {'任务id':'','status':''}
        :param key:  列表键名
        :param value:   列表值   {'任务id':'','status':''}
        :return:
        """
        try:
            if isinstance(value, dict):
                value = json.dumps(value)
            lock = redis_lock.Lock(self.redis_cli, key, expire=15)
            with lock:
                self.redis_cli.rpush(key, value)
            return 1
        except Exception as e:
            print(e)
            return 0

    def redis_hincrbyfloat(self, _hash_name: str, key: str, num: int):
        """  哈希
        将值  进行增加
        :param _hash_name:  hash 名称
        :param key:
        :param num:  增加的数量
        :return:
        """

        lock = redis_lock.Lock(self.redis_cli, _hash_name, expire=15)
        with lock:
            aa = self.redis_cli.hincrbyfloat(_hash_name, key, num)
        return aa

    def redis_zset(self, key: str, type_: str, member: str, num=0):
        """   redis 有序集合
        :param key:             redis 键值
        :param type_:           调用类型   1:新增或者更新新数据  2:分数自增 3:得到分数
        :param member:          有序集合的 成员名称
        :param num:             分数值  或者自增值
        :return:
        """
        lock = redis_lock.Lock(self.redis_cli, key, expire=15)
        with lock:
            if type_ == '1':
                zadd = self.redis_cli.zadd(key, {member: num})
            elif type_ == '2':
                zadd = self.redis_cli.zincrby(key, num, member)
            elif type_ == '3':
                zadd = self.redis_cli.zscore(key, member)
        if zadd:
            return round(zadd, 1)

    def redis_delete(self, key):
        """
        删除键的值
        :param key:
        :return:
        """
        self.redis_cli.delete(key)

    def redis_delete_hash(self, _hash_name: str, key):
        """  哈希
        删除哈希
        :param _hash_name:
        :return:
        """
        self.redis_cli.hdel(_hash_name, key)

    def hget(self, _hash_name: str, key: str):
        """  哈希
        获取健的值
        :param key:  键的名称
        :return: 值  类型 dict
        """
        return self.redis_cli.hget(_hash_name, key)


if __name__ == '__main__':
    redis_ = RedisWithLock()
    # get = redis_.redis_get('hash_name_state', 'csk-316-248/usual/111710815014926')
    # print(get)

    redis_.record_status('cj_status_info', json.dumps({'task_id': 'id', 'state': '10'}))
    # print(redis_.redis_zset('his', 'weee', 1))
    # print(redis_.redis_delete('hash_name_state'))
    # get = redis_.redis_get('hash_name_state', '1705376847')
    # print(get,type(get))
    # redis_.redis_update('test1', {'start': 12,'a':21})
    # print(redis_.redis_get('test1'))
